/*******************************************************************
 * Copyright 2002-2023 Accusoft Corporation, Tampa Florida.        *
 * This sample code is provided to Accusoft licensees "as is"      *
 * with no restrictions on use or modification. No warranty for    *
 * use of this sample code is provided by Accusoft.                *
 *******************************************************************
 * BarcodeXpress writebarcodes sample: 
 * This sample program demonstrates the writer API for BarcodeXpress: 
 *   - It writes 6 different barcode images to the current directory. 1 each of the 
 *     supported 2D barcodes and 1 each of common 1D barcode types. 
 *   - In addition, it demonstrates using the reader API to read and confirm the value of the
 *     newly created barcode.
 */

#include "barcodexpress.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

void create_barcode_file(BX_CreateParameters* params, const char* outputfilename); 
void create_barcode_dib(BX_CreateParameters* params, void** dibPointer); 
void create_2D_barcodes(); 
void create_1D_barcodes(); 
void readback_barcode_file(const char* inputFileName, const char* expectedValue);
void readback_barcode_dib(void** dibPointer, const char* expectedValue);
void setupLicense(); 

int main(int argc, char** argv) 
{
    setupLicense();
    create_2D_barcodes(); 
    create_1D_barcodes(); 
    
    printf("FINISHED\n"); 
    return (EXIT_SUCCESS);
}

void setupLicense( ) {
    // Set the environment variable for the location of Accusoft licenses.
    // If this variable is not set, then the license file must be located in one of the
    // standard locations as specified in the help file.
    // char *licensePath = "/custom/path/to/license/file";
    // setenv( "ACCUSOFT_LICENSE_DIR", licensePath, 1 );

    /*
    * Set Barcode Xpress Runtime Licensing:
    * These methods should only be called if this application is deployed on a machine with a runtime license installed.
    * See the help file for information about how to install a runtime license.
    * If this is a development machine, then there is no need to call the licensing functions.
    * An evaluation or development license will be used.
    */

    // BX_set_solution_name("YourSolutionName");
    // BX_set_solution_key(key1, key2, key3, key4);
    // BX_set_oem_license_key("2.0.AStringForOEMLicensingContactAccusoftSalesForMoreInformation...");
}

void create_barcode_dib(BX_CreateParameters* params, void** dib)
{
    BX_CreateResult result;
    BX_Status status = BX_Status_OK;
    status = BX_create_dib(params, dib, &result);

    if( BX_Status_OK != status ) {
        printf("\tERRORS:\n");
        for( int i=0; i<result.ErrorCount; ++i ) {
            BX_Error *error = &(result.BarcodeErrors[i]);
            printf("\t\t%d: %s\n\t%s\n", error->ErrorStatus, error->ErrorID, error->ErrorMessage);
    }
    } else {
        printf("Successfully created dib.\n");
        }
    }

void create_barcode_file(BX_CreateParameters* params, const char* fileName)
{
    BX_CreateResult result;
    BX_Status status = BX_Status_OK;
    status = BX_create_file(params, fileName, &result);

    if( BX_Status_OK != status ) {
        printf("\tERRORS:\n");
        for( int i=0; i<result.ErrorCount; ++i ) {
            BX_Error *error = &(result.BarcodeErrors[i]);
            printf("\t\t%d: %s\n\t%s\n", error->ErrorStatus, error->ErrorID, error->ErrorMessage);
    }
    } else {
        printf("Successfully created file \"%s\".\n", fileName);
    }
}

void create_2D_barcodes()
{
    const char* dibFile0 = "PDF417_Barcode.bmp";
    const char* dibFile1 = "QRCode_Barcode.bmp";
    const char* dibFile2 = "DataMatrix_Barcode.bmp";
    const char* v1 = "An example PDF417 barcode ";
    const char* v2 = "An example QRCode barcode";
    const char* v3 = "An example DataMatrix barcode";
    void* dibPointer = NULL; 
    BX_CreateParameters bxCreateParams = BX_DefaultCreateParameters; 

    BX_BarcodeType types[] = {BX_BarcodeType_PDF417, BX_BarcodeType_QRCode, BX_BarcodeType_DataMatrix}; 
    const char* fileNames[] = {dibFile0, dibFile1, dibFile2}; 
    const char* bxValues[] = {v1, v2, v3}; 
    
    for (int i = 0; i < 3; i++)
    {
        bxCreateParams.BarcodeType = types[i]; 
        bxCreateParams.Value = bxValues[i]; 
        bxCreateParams.ValueLength = strlen(bxValues[i]);

        // Create and verify the file barcode
        create_barcode_file(&bxCreateParams, fileNames[i]);
        readback_barcode_file(fileNames[i], bxCreateParams.Value);

        // Create and verify the dib barcode
        create_barcode_dib(&bxCreateParams, &dibPointer);
        readback_barcode_dib(&dibPointer, bxCreateParams.Value);

        // Free memory allocated to dib
        BX_free_dib(&dibPointer);
    }
}

void create_1D_barcodes()
{
    const char* dibFile0 = "UPCA_Barcode.bmp"; 
    const char* dibFile1 = "Code39_Barcode.bmp"; 
    const char* dibFile2 = "Code128_Barcode.bmp"; 
    const char* v1 = "439199298873"; // The last character value is the checksum. If this is not correct, it will be replaced. 
                                    //  If it is not present then it will automatically be calculated and added. 
    const char* v2 = "3+9ISCODE39"; 
    const char* v3 = "Code128WritesAllASCII"; 
    void * dibPointer = NULL; 
    BX_BarcodeType types[] = {BX_BarcodeType_UPCA, BX_BarcodeType_Code39, BX_BarcodeType_Code128}; 
    const char* fileNames[] = {dibFile0, dibFile1, dibFile2}; 
    const char* bxValues[] = {v1, v2, v3}; 
    BX_CreateParameters bxCreateParams = BX_DefaultCreateParameters; 
    // Set some of the 1D parameters
    bxCreateParams.Params1D.CheckSum = false; // Don't print the checksum for barcode types that have an optional checksum character.
    bxCreateParams.Params1D.TextNotchPercent = 25; 
    
    // Write the barcodes
    for (int i = 0; i < 3; i++)
    {
        bxCreateParams.Params1D.ShowText = (BX_ShowTextInfo)i; //demonstrate the different text views.
        bxCreateParams.BarcodeType = types[i]; 
        bxCreateParams.Value = bxValues[i]; 
        bxCreateParams.ValueLength = strlen(bxValues[i]);
        
        // Create and verify the file barcode
        create_barcode_file(&bxCreateParams, fileNames[i]); 
        readback_barcode_file(fileNames[i], bxCreateParams.Value);

        // Create and verify the dib barcode
        create_barcode_dib(&bxCreateParams, &dibPointer); 
        readback_barcode_dib(&dibPointer, bxCreateParams.Value);

        // Free memory allocated to dib
        BX_free_dib(&dibPointer); 
    }
}

void readback_barcode_file(const char* inputFileName, const char* expectedValue)
{
    BX_Status status = BX_Status_OK;
 
    // Set analyze parameters: 
    BX_AnalyzeParameters ps = BX_DefaultAnalyzeParameters; 
    // Only search for the types we created:
    ps.BarcodeTypes = (BX_BarcodeType)(BX_BarcodeType_QRCode | 
            BX_BarcodeType_PDF417 | 
            BX_BarcodeType_DataMatrix |
            BX_BarcodeType_Code128 |
            BX_BarcodeType_Code39 |
            BX_BarcodeType_UPCA);  
    BX_AnalyzeResult result = {0};
    ps.Orientation = BX_Orientation_HorizontalVerticalDiagonal;

    status = BX_analyze_file(&ps, inputFileName, &result);
    
    printf("\tWrite value: %s\n", expectedValue);
    if( BX_Status_OK != status ) {
        printf("\tERRORS:\n");
        for( int i=0; i<result.ErrorCount; ++i ) {
            BX_Error *error = &(result.BarcodeErrors[i]);
            printf("\t\t%d: %s\n\t%s\n", error->ErrorStatus, error->ErrorID, error->ErrorMessage);
        }
        return;
        }        

    for(int i=0; i<result.BarcodeResultsLength; ++i) {
        BX_BarcodeResult *res = &result.BarcodeResults[i];
        printf("\tRead value:  %s\n", res->Value);
    }
    
    // Free up resources
    BX_free_analyze_result(&result);
}

void readback_barcode_dib(void** dibPointer, const char* expectedValue)
{
    BX_Status status = BX_Status_OK;
    // Set analyze parameters
    BX_AnalyzeParameters ps = BX_DefaultAnalyzeParameters; 
    // Only search for the types we created
    ps.BarcodeTypes = (BX_BarcodeType)(BX_BarcodeType_QRCode | 
            BX_BarcodeType_PDF417 | 
            BX_BarcodeType_DataMatrix |
            BX_BarcodeType_Code128 |
            BX_BarcodeType_Code39 |
            BX_BarcodeType_Code39Ext |
            BX_BarcodeType_UPCA);  
    BX_AnalyzeResult result = {0};
    ps.Orientation = BX_Orientation_HorizontalVerticalDiagonal;

//    Uncomment the following line to see the effect this has for the code 128. 
//    ps.AppendCheckSum = true; 
    status = BX_analyze_dib(&ps, *dibPointer, &result);
    
    printf("\tWrite value: %s\n", expectedValue);
    if( BX_Status_OK != status ) {
        printf("\tERRORS:\n");
        for( int i=0; i<result.ErrorCount; ++i ) {
            BX_Error *error = &(result.BarcodeErrors[i]);
            printf("\t\t%d: %s\n\t%s\n", error->ErrorStatus, error->ErrorID, error->ErrorMessage);
        }
        return;
        }        

    for(int i=0; i<result.BarcodeResultsLength; ++i) {
        BX_BarcodeResult *res = &result.BarcodeResults[i];
        printf("\tRead value:  %s\n", res->Value);
    }

    // Free up resources
    BX_free_analyze_result(&result);
}
