###################################################################
# Copyright 2002-2025 Accusoft Corporation, Tampa Florida.        #
# This sample code is provided to Accusoft licensees "as is"      #
# with no restrictions on use or modification. No warranty for    #
# use of this sample code is provided by Accusoft.                #
###################################################################
""" 
BarcodeExpress.py - Python wrapper for libbarcodexpress 

This module implements a pythonic wrapper for libbarcodexpress
from BarcodeXpress Linux. The following classes and functions are
considered essential:

    AnalyzeFile - Loads a bmp file from disk and recognizes barcodes.
    AnalyzeOpenCV - Accepts an opencv image and recognizes barcodes.

    Barcode - Represents a detected barcode and contains all relevant values.
"""

import ctypes
from pathlib import Path
from os import path

# Version info
major_version = "14"
minor_version = "0"
__version__ = "0"

# The following set of classes implement the structures from barcodexpress.h
# which are needed for analyze calls.
class _CRectangle(ctypes.Structure):
    _fields_ = [
        ("Left",   ctypes.c_long),
        ("Top",    ctypes.c_long),
        ("Right",  ctypes.c_long),
        ("Bottom", ctypes.c_long)
    ]

class _CPoint(ctypes.Structure):
    _fields_ = [
        ("X", ctypes.c_long),
        ("Y", ctypes.c_long)
    ]

class _CAnalyzeParameters(ctypes.Structure):
    _fields_ = [
        ("ParametersVersion",               ctypes.c_long),
        ("AppendCheckSum",                  ctypes.c_bool),
        ("Area",                            _CRectangle),
        ("BarcodeTypes",                    ctypes.c_ulonglong),
        ("IncludeControlCharacters",        ctypes.c_bool),
        ("InkColor",                        ctypes.c_int),
        ("MaximumBarcodes",                 ctypes.c_long),
        ("MinimumBarcodeSize",              ctypes.c_long),
        ("Orientation",                     ctypes.c_int),
        ("ReturnPossibleBarcodes",          ctypes.c_bool),
        ("AustralianPostDecodeType",        ctypes.c_int),
        ("RoyalMailVariableLengthDecoding", ctypes.c_bool),
        ("ScanDistance",                    ctypes.c_int),
        ("GreyscaleProcessing",             ctypes.c_bool),
        ("AdditionalReadingPass",           ctypes.c_bool),
        ("DebugLogger",                     ctypes.c_void_p)
    ]

class _CBarcodeResult2D(ctypes.Structure):
    _fields_ = [
        ("RowsDetected", ctypes.c_long),
        ("ColumnsDetected", ctypes.c_long),
        ("Rows", ctypes.c_long),
        ("Columns", ctypes.c_long),
        ("ErrorCorrectionLevel", ctypes.c_long)
    ]

class _CBarcodeResult(ctypes.Structure):
    _fields_ = [
        ("Area", _CRectangle),
        ("Point1", _CPoint),
        ("Point2", _CPoint),
        ("Point3", _CPoint),
        ("Point4", _CPoint),
        ("Confidence", ctypes.c_long),
        ("Name", ctypes.c_char_p),
        ("NameLength", ctypes.c_long),
        ("Skew", ctypes.c_long),
        ("BarcodeType", ctypes.c_ulonglong),
        ("CheckSumValid", ctypes.c_bool),
        ("CheckSumLength", ctypes.c_long),
        ("Value", ctypes.c_char_p),
        ("ValueLength", ctypes.c_long),
        ("ModeTransitions", ctypes.POINTER(ctypes.c_int)),
        ("ModeTransitionsLength", ctypes.c_long),
        ("Result2D", _CBarcodeResult2D)
    ]

class _CBarcodeError(ctypes.Structure):
    _fields_ = [
        ("ErrorStatus", ctypes.c_int),
        ("ErrorID", ctypes.c_char_p),
        ("ErrorMessage", ctypes.c_char_p)
    ]

class _CAnalyzeResult(ctypes.Structure):
    _fields_ = [
        ("BarcodeResults", ctypes.POINTER(_CBarcodeResult)),
        ("BarcodeErrors", _CBarcodeError * 64),
        ("BarcodeResultsLength", ctypes.c_long),
        ("ErrorCount", ctypes.c_long)
    ]

def _FindNativeLibrary():
    lib_locations = [
        Path("."),
        Path("../../bin"),
        Path(path.expanduser("~") + "/Accusoft/BarcodeXpress" + major_version + "-64/bin")
    ]

    lib_extensions = [
        "." + major_version + "." + minor_version,
        "." + major_version,
        ""
    ]

    for location in lib_locations:
        for extension in lib_extensions:
            lib_path = location / ("libbarcodexpress.so" + extension)
            if lib_path.exists():
                return lib_path

    raise BarcodeError(None, None, "Unable to locate libbarcodexpress.")

class _Status:
    # The static members below coorespond to the values of the BX_Status
    # enum in barcodexpress.h and are presented here for convenience.
    # This class also contains a Messages member which maps the integer
    # values back to the names so that they can be printed in the case that
    # an error occurs.
    OK = 0
    OutOfMemory = 9000
    InvalidParameter = 9001
    ParameterOutOfRange = 9002
    InvalidBitmap = 9003
    UnableToReadBitmap = 9004
    UnsupportedBarcodeType = 9005
    InvalidIndex = 9006
    InvalidBarcodeType = 9007
    ImageWidthTooSmall = 9008
    ImageHeightTooSmall = 9009
    UnableCreateUPCE = 9010
    InvalidLicense = 9011
    Timeout = 9012
    InternalError = 9014
    InvalidArea = 9015
    UnsupportedBitDepth = 9022
    NullResults = 9023
    InvalidDIB = 9024
    PDF417EncodingError = 9100
    InvalidRowCount = 9101
    InvalidColumnCount = 9102
    SymbolCountError = 9103
    ValueTooLarge = 9104
    InvalidErrorCorrection = 9105
    ErrorCorrectionLevelTooHigh = 9106
    UnableToReadParameters = 9300
    UnableToReadDIB = 9301
    InvalidResults = 9302
    UnableToCreateResults = 9303
    NoLicenseFound = 9304
    EncodeError = 9305

    Messages = {
        # Maps the integer values back to the names so that they can be printed
        # in the case that an error occurs.
        0: "OK",
        9000: "OutOfMemory",
        9001: "InvalidParameter",
        9002: "ParameterOutOfRange",
        9003: "InvalidBitmap",
        9004: "UnableToReadBitmap",
        9005: "UnsupportedBarcodeType",
        9006: "InvalidIndex",
        9007: "InvalidBarcodeType",
        9008: "ImageWidthTooSmall",
        9009: "ImageHeightTooSmall",
        9010: "UnableCreateUPCE",
        9011: "InvalidLicense",
        9012: "Timeout",
        9014: "InternalError",
        9015: "InvalidArea",
        9022: "UnsupportedBitDepth",
        9023: "NullResults",
        9024: "InvalidDIB",
        9100: "PDF417EncodingError",
        9101: "InvalidRowCount",
        9102: "InvalidColumnCount",
        9103: "SymbolCountError",
        9104: "ValueTooLarge",
        9105: "InvalidErrorCorrection",
        9106: "ErrorCorrectionLevelTooHigh",
        9300: "UnableToReadParameters",
        9301: "UnableToReadDIB",
        9302: "InvalidResults",
        9303: "UnableToCreateResults",
        9304: "NoLicenseFound",
        9305: "EncodeError"
    }

class BarcodeType:
    """
    BarcodeXpress.BarcodeType - Static class which contains BX Barcode Types

    Analyze Functions in this module accept a list of integers which 
    coorespond to barcode types. This class contains a number of static members 
    which contain the relevant integers:

        Unknown, Industrial2of5, Interleaved2of5, IATA2OF5, Datalogic2of5, 
        Inverted2of5, BCDMatrix, Matrix2of5, Code32, Code39, Codabar, Code93, 
        Code128, EAN13, EAN8, UPCA, UPCE, Add5, Add2, EAN128, PatchCode,
        Planet, PostNet, PDF417, DataMatrix, Code39Ext, Code93Ext, QRCode, 
        IntelligentMail, RoyalPost4State, AustralianPost4State, Aztec, 
        GS1Databar, UPU4State, MicroPDF417 
            
    Also included are three composite members which represent multiple barcode
    types simultaneously:

        All1D, All2D, All

    Any single barcode type or a combination of barcode types can be searched for.
    If you specify multiple barcodes types and one of them is Unknown then only 
    the explicity specified types will be searched for, and not all the other
    types included in Unknown. The same applies to PatchCode. If you
    search for PatchCode and any other type, only the other types will be searched
    for. PatchCode must be the only specified type if you wish to search for them.

    This class also includes a Names member which maps the barcode type names to
    the integers so that they can be looked up from their names programmatically.
    """
    Unknown = 0
    Industrial2of5        = 1 << 0
    Interleaved2of5       = 1 << 1
    IATA2OF5              = 1 << 2
    Datalogic2of5         = 1 << 3
    Inverted2of5          = 1 << 4
    BCDMatrix             = 1 << 5
    Matrix2of5            = 1 << 6
    Code32                = 1 << 7
    Code39                = 1 << 8
    Codabar               = 1 << 9
    Code93                = 1 << 10
    Code128               = 1 << 11
    EAN13                 = 1 << 12
    EAN8                  = 1 << 13
    UPCA                  = 1 << 14
    UPCE                  = 1 << 15
    Add5                  = 1 << 16
    Add2                  = 1 << 17
    EAN128                = 1 << 18
    PatchCode             = 1 << 19
    PostNet               = 1 << 20
    PDF417                = 1 << 21
    DataMatrix            = 1 << 22
    Code39Ext             = 1 << 23
    Code93Ext             = 1 << 24
    QRCode                = 1 << 25
    IntelligentMail       = 1 << 26
    RoyalPost4State       = 1 << 27
    AustralianPost4State  = 1 << 28
    Aztec                 = 1 << 29
    GS1Databar            = 1 << 30
    UPU4State             = 1 << 31
    MicroPDF417           = 1 << 32
    Planet                = 1 << 33
    MicroQRCode           = 1 << 34

    # Special Combined types
    All1D = (Add2 | Add5 | AustralianPost4State | BCDMatrix | Codabar |
             Code32 | Code39 | Code39Ext | Code93 | Code93Ext | Code128 |
             Datalogic2of5 | EAN8 | EAN13 | EAN128 | GS1Databar |
             IATA2OF5 | Industrial2of5 | IntelligentMail |
             Interleaved2of5 | Inverted2of5 | Matrix2of5 | PatchCode |
             Planet | PostNet | RoyalPost4State | UPCA | UPCE | UPU4State)

    All2D = (Aztec | DataMatrix | PDF417 | MicroPDF417 | QRCode | MicroQRCode)

    All = (All1D | All2D)

    Names = {
        # Maps the barcode type names to thier integer values
        # so that they can be looked up programatically.
        "Unknown":               0,
        "Industrial2of5":        1 << 0,
        "Interleaved2of5":       1 << 1,
        "IATA2OF5":              1 << 2,
        "Datalogic2of5":         1 << 3,
        "Inverted2of5":          1 << 4,
        "BCDMatrix":             1 << 5,
        "Matrix2of5":            1 << 6,
        "Code32":                1 << 7,
        "Code39":                1 << 8,
        "Codabar":               1 << 9,
        "Code93":                1 << 10,
        "Code128":               1 << 11,
        "EAN13":                 1 << 12,
        "EAN8":                  1 << 13,
        "UPCA":                  1 << 14,
        "UPCE":                  1 << 15,
        "Add5":                  1 << 16,
        "Add2":                  1 << 17,
        "EAN128":                1 << 18,
        "PatchCode":             1 << 19,
        "PostNet":               1 << 20,
        "PDF417":                1 << 21,
        "DataMatrix":            1 << 22,
        "Code39Ext":             1 << 23,
        "Code93Ext":             1 << 24,
        "QRCode":                1 << 25,
        "IntelligentMail":       1 << 26,
        "RoyalPost4State":       1 << 27,
        "AustralianPost4State":  1 << 28,
        "Aztec":                 1 << 29,
        "GS1Databar":            1 << 30,
        "UPU4State":             1 << 31,
        "MicroPDF417":           1 << 32,
        "Planet":                1 << 33,
        "MicroQRCode":           1 << 34
    }

class InkColor:
    """ 
    Valid ink colors which can be passed as options to analyze calls.
    
    The valid colors are: Unknown, Black, and White
    """
    Unknown = 0
    Black = 1
    White = 2

class Orientation:
    """
    Valid orientations which can be passed as options to analyze calls.
    """
    HorizontalVertical = 0
    Horizontal = 1
    Vertical = 2
    HorizontalVerticalDiagonal = 3

class ModeTransitionType:
    """
    Specifies the types of possible state transitions in the barcode
    reader's internal state. Cooresponds to the BX_ModeTransitionType
    enum and it's values in barcodexpress.h.        
    """
    NoModeTransition = 0
    QRNumeric = 1
    QRAlpha = 2
    QRByte = 3
    QRKanji = 4
    QRECI = 5
    QRFNC1First = 6
    QRFNC1Second = 7
    QRStructureAppend = 8

class Barcode:
    """
    Represents a barcode detected by an analyze call and contains the 
    following fields which correspond to the values of the BarcodeResult
    and BarcodeResult2D structs in barcodexpress.h:

        Area - Specifies a rectangular area on the image containing the 
        barcode. This area does not necessarily correspond to the corner 
        points of a barcode.

        Point1 - Specifies the upper-left corner point of the barcode.
        Point2 - Specifies the upper-right corner point of the barcode.
        Point3 - Specifies the lower-right corner point of the barcode.
        Point4 - Specifies the lower-left corner point of the barcode.

        Confidence - Specifies the confidence value that the barcode 
                     result is accurate. Confidence values of 30 indicate 
                     that the barcode value could not be determined accurately.

        Name - This is a string representation of the barcode-type.

        Skew - Specifies the angle of skew for the recognized barcode. The 
               skew angle is measured in degrees, with 0 degrees being 
               horizontal. It is measured from the horizontal axis to the top 
               edge of the barcode.

        Type - Specifies the type of barcode that is detected.

        CheckSumValid - Specifies if the checksum for the barcode is valid.

        Value - Specifies a string containing the value of a recognized barcode

        ModeTransitions - Specifies the Mode Transitions for barcodes that 
                          support it.

        The following fields are only relevant for 2D Barcodes:

        RowsDetected - Specifies the number of rows detected in the barcode 
                       image.

        ColumnsDetected - Specifies the number of columns detected in the 
                          barcode image.

        Rows - Specifies the number of rows in the barcode as defined by the 
               indicator pattern.

        Columns - Specifies the number of columns in the barcode as defined 
                  by the indicator pattern. Column Count for PDF417 barcodes 
                  is the number of codeword columns.

        ErrorCorrectionLevel - Specifies the amount of error correction 
                               detected for QR and PDF417 barcodes.
    """
    def __init__(self, barcode):
        self.Area            = Rectangle.FromCRectangle(barcode.Area)
        self.Point1          = Point.FromCPoint(barcode.Point1)
        self.Point2          = Point.FromCPoint(barcode.Point2)
        self.Point3          = Point.FromCPoint(barcode.Point3)
        self.Point4          = Point.FromCPoint(barcode.Point4)
        self.Confidence      = barcode.Confidence
        self.Name            = barcode.Name.decode("utf-8")
        self.Skew            = barcode.Skew
        self.Type            = int(barcode.BarcodeType)
        self.CheckSumValid   = barcode.CheckSumValid
        self.Value           = barcode.Value.decode("utf-8")
        self.ModeTransitions = [barcode.ModeTransitions[i] for i in range(0, barcode.ModeTransitionsLength)]

        result2D = barcode.Result2D

        self.RowsDetected    = result2D.RowsDetected
        self.Rows            = result2D.Rows
        self.ColumnsDetected = result2D.ColumnsDetected
        self.Columns         = result2D.Columns
        self.ErrorCorrectionLevel = result2D.ErrorCorrectionLevel
        
class Rectangle:
    """Describes a bounding area with Top, Bottom, Left, and Right fields."""
    def __init__(self, top, bottom, left, right):
        self.Top = top
        self.Bottom = bottom
        self.Left = left
        self.Right = right

    @classmethod
    def FromCRectangle(cls, cRect):
        # Use this to make these from the structures returned by BX.
        return Rectangle(cRect.Top, cRect.Bottom, cRect.Left, cRect.Right)

class Point:
    """Describes a 2D Point with X and Y fields."""
    def __init__(self, x, y):
        self.X = x
        self.Y = y

    @classmethod
    def FromCPoint(cls, cPoint):
        # Use this to make these from the structures returned by BX.
        return Point(cPoint.X, cPoint.Y)

class BarcodeError(Exception):
    """Non fatal errors returned from Barcode Xpress"""
    def __init__(self, status = None, bxerror = None, custom_message = None):
        if status != None:
            self.Status = status
        else:
            self.Status = 0
        
        self.CustomMessage = custom_message

        if bxerror != None:
            self.ID = bxerror.ErrorID.decode("utf-8")
            self.Message = bxerror.ErrorMessage.decode("utf-8")
        else:
            self.ID = ""
            if status in _Status.Messages:
                self.Message = _Status.Messages[status]
            else:
                self.Message = "Unknown Error"

    def __str__(self):
        if self.CustomMessage != None:
            return self.CustomMessage

        return "BarcodeError(" + str(self.Status) + ": " + str(self.ID) + " " + self.Message + ")"
        
        
def _DefaultAnalyzeParameters():
    # Creates default parameters. Only used internally.
    return _CAnalyzeParameters(
        ParametersVersion = 1,
        AppendCheckSum = False,
        Area = _CRectangle(0, 0, 0, 0),
        BarcodeTypes = BarcodeType.Unknown,
        IncludeControlCharacters = False,
        InkColor = 1, # Black
        MaximumBarcodes = 100,
        MinimumBarcodeSize = 0,
        Orientation = 0, # HorizontalVertical
        ReturnPossibleBarcodes = False,
        AustralianPostDecodeType = 3, # NoCustomDecode
        RoyalMailVariableLengthDecoding = False,
        ScanDistance = 5,
        GreyscaleProcessing = False,
        AdditionalReadingPass = False,
        DebugLogger = None
    )

def _CreateAnalyzeParameters(options):
    # This function accepts a dictionary of analyze parameters
    # and fills out the AnalyzeParameters struct with
    # any values provided.
    parameters = _DefaultAnalyzeParameters()

    if "AppendCheckSum" in options:
        parameters.AppendCheckSum = options["AppendCheckSum"]
    if "Area" in options:
        parameters.Area = _CRectangle(
            Top = options["Area"].Top,
            Bottom = options["Area"].Bottom,
            Left = options["Area"].Left,
            Right = options["Area"].Right
        )
    if "BarcodeTypes" in options:
        bctypes = 0
        for bctype in options["BarcodeTypes"]:
            bctypes = bctypes | bctype
        parameters.BarcodeTypes = bctypes
    if "IncludeControlCharacters" in options:
        parameters.IncludeControlCharacters = options["IncludeControlCharacters"]
    if "InkColor" in options:
        parameters.InkColor = options["InkColor"]
    if "MaximumBarcodes" in options:
        parameters.MaximumBarcodes = options["MaximumBarcodes"]
    if "MinimumBarcodeSize" in options:
        parameters.MinimumBarcodeSize = options["MinimumBarcodeSize"]
    if "Orientation" in options:
        parameters.Orientation = options["Orientation"]
    if "ReturnPossibleBarcodes" in options:
        parameters.ReturnPossibleBarcodes = options["ReturnPossibleBarcodes"]
    if "AustralianPostDecodeType" in options:
        parameters.AustralianPostDecodeType = options["AustralianPostDecodeType"]
    if "RoyalMailVariableLengthDecoding" in options:
        parameters.RoyalMailVariableLengthDecoding = options["RoyalMailVariableLengthDecoding"]
    if "ScanDistance" in options:
        parameters.ScanDistance = options["ScanDistance"]
    if "GreyscaleProcessing" in options:
        parameters.GreyscaleProcessing = options["GreyscaleProcessing"]
    if "AdditionalReadingPass" in options:
        parameters.AdditionalReadingPass = options["AdditionalReadingPass"]

    return parameters

def AnalyzeFile(filename, options = {}):
    """
    Detects barcodes from a BMP image located on disk and returns
    a list of Barcode objects.

    This function accepts a filename and an options dictionary which
    recognizes all parameters of BX_AnalyzeParameters in 
    barcodexpress.h. Most of those parameters are either integers or
    boolean, but the following notes apply to those which are not:

    Area - Use the Rectangle class provided in this module.

    BarcodeTypes - Provide a python list of the barcode type integers.

    For the following options, classes are provided in this module which
    contain the valid values.
    
        InkColor
        Orientation
        AustralianPostDecodeType

    Example: 

        results = BarcodeXpress.AnalyzeFile("../images/Barcode-All-Supported-Types.bmp", {
            "Area": BarcodeXpress.Rectangle(0, 0, 0, 0),
            "BarcodeTypes": [BarcodeXpress.BarcodeType.Code128, BarcodeExpress.BarcodeType.MicroPDF417],
            "Orientation": BarcodeXpress.Orientation.HorizontalVertical
        })

    """
    parameters = _CreateAnalyzeParameters(options)
    result = _CAnalyzeResult()

    status = BX.BX_analyze_file(ctypes.byref(parameters), filename.encode('utf-8'), ctypes.byref(result))

    barcodes_found = []

    if status == _Status.OK:
        barcodes_found = [Barcode(result.BarcodeResults[i]) for i in range(0, result.BarcodeResultsLength)]

    BX.BX_free_analyze_result(ctypes.byref(result))

    if result.ErrorCount > 0:
        raise BarcodeError(status, result.BarcodeErrors[0])

    if status != _Status.OK:
        print("Error Status Returned from BX: " + _Status.Messages[status])
        
    return barcodes_found

def AnalyzeOpenCV(image, options= {}):
    """
    Detects barcodes from an opencv image located in memory and returns
    a list of Barcode objects.

    This function accepts an opencv image and an options dictionary which
    recognizes all parameters of BX_AnalyzeParameters in 
    barcodexpress.h. Most of those parameters are either integers or
    boolean, but the following notes apply to those which are not:

    Area - Use the Rectangle class provided in this module.

    BarcodeTypes - Provide a python list of the barcode type integers.

    For the following options, classes are provided in this module which
    contain the valid values.
    
        InkColor
        Orientation
        AustralianPostDecodeType

    Example: 

        results = BarcodeXpress.AnalyzeOpenCV(opencv_image, {
            "Area": BarcodeXpress.Rectangle(0, 500, 0, 500),
            "BarcodeTypes": [BarcodeXpress.BarcodeType.All1D],
            "Orientation": BarcodeXpress.Orientation.Horizontal
        })

    """
    import cv2, numpy

    parameters = _CreateAnalyzeParameters(options)
    result = _CAnalyzeResult()

    _, bmp = cv2.imencode(".bmp", image)

    status = BX.BX_analyze_bmp(ctypes.byref(parameters), bmp.ctypes.data_as(ctypes.POINTER(ctypes.c_uint8)), len(bmp), ctypes.byref(result))

    barcodes_found = []

    if result.ErrorCount > 0:
        raise BarcodeError(status, result.BarcodeErrors[0])

    if status != _Status.OK:
        print("Error Status Returned from BX: " + _Status.Messages[status])
    else:
        barcodes_found = [Barcode(result.BarcodeResults[i]) for i in range(0, result.BarcodeResultsLength)]

    BX.BX_free_analyze_result(ctypes.byref(result))

    return barcodes_found

def SetOEMLicenseKey(key):
    """
    SetOEMLicenseKey - Sets the BarcodeXpress control runtime license deployment key.

    The OEM key needs to be set only if purchased. If you have not purchased an
    OEM key then do not call this function.
    """
    return BX.BX_set_oem_license_key(key.encode("utf-8"))

def SetSolutionName(name):
    """Sets the Barcode Xpress control runtime license deployment solution name."""
    return BX.BX_set_solution_name(name.encode("utf-8"))

def SetSolutionKey(key1, key2, key3, key4):
    """Sets the BarcodeXpress control runtime license deployment solution keys."""
    return BX.BX_set_solution_key(key1, key2, key3, key4)


# Load the shared library. Only works in Linux for now.
library_path = _FindNativeLibrary()
BX = ctypes.cdll.LoadLibrary(library_path)

# Set parameter types for the needed functions.
BX.BX_set_solution_name.argtypes = [ctypes.c_char_p]
BX.BX_set_solution_key.argtypes = [ctypes.c_long, ctypes.c_long, ctypes.c_long, ctypes.c_long]
BX.BX_set_oem_license_key.argtypes = [ctypes.c_char_p]
BX.BX_analyze_file.argtypes = [ctypes.POINTER(_CAnalyzeParameters), ctypes.c_char_p, ctypes.POINTER(_CAnalyzeResult)]
BX.BX_analyze_bmp.argtypes = [ctypes.POINTER(_CAnalyzeParameters), ctypes.POINTER(ctypes.c_uint8), ctypes.c_ulong, ctypes.POINTER(_CAnalyzeResult)]
BX.BX_free_analyze_result.argtypes = [ctypes.POINTER(_CAnalyzeResult)]





