/*******************************************************************
 * Copyright 2002-2023 Accusoft Corporation, Tampa Florida.        *
 * This sample code is provided to Accusoft licensees "as is"      *
 * with no restrictions on use or modification. No warranty for    *
 * use of this sample code is provided by Accusoft.                *
 *******************************************************************
 * BarcodeXpress readbarcodesig sample:
 * This sample program demonstrates the reader API for BarcodeXpress:
 *   - It reads each image file passed in to the command line, after
 *     being opened & converted by ImageGear.
 */

#include "barcodexpress.h"
#include "stdlib.h"
#include "stdio.h"

#include <gear.h>
#include <gearestr.h>
#include <linux/limits.h>
#include <sys/stat.h>

#include <sys/wait.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

void setupLicense( ) {
    // Set the environment variable for the location of Accusoft licenses.
    // If this variable is not set, then the license file must be located in one of the
    // standard locations as specified in the help file.
    // char *licensePath = "/custom/path/to/license/file";
    // setenv( "ACCUSOFT_LICENSE_DIR", licensePath, 1 );

    /*
    * Set Barcode Xpress Runtime Licensing:
    * These methods should only be called if this application is deployed on a machine with a runtime license installed.
    * See the help file for information about how to install a runtime license.
    * If this is a development machine, then there is no need to call the licensing functions.
    * An evaluation or development license will be used.
    */

    // BX_set_solution_name("YourSolutionName");
    // BX_set_solution_key(key1, key2, key3, key4);
    // BX_set_oem_license_key("2.0.AStringForOEMLicensingContactAccusoftSalesForMoreInformation...");
}

void setupImageGear( ) {
	AT_ERRCOUNT nErrCount = IG_initialize(NULL, 0, FALSE);
	if(nErrCount!=0)
	{
		printf("Error on IG init\n");
		exit(1);
	}

	// License ImageGear
	// IG_lic_solution_name_set ("YourSolutionName");
	// IG_lic_solution_key_set (key1, key2, key3, key4);
	// IG_lic_OEM_license_key_set("2.0.AStringForOEMLicensingContactAccusoftSalesForMoreInformation..."); 
}

void ImageGearErrOut() {
    CHAR         error_not_found[] = "No error description";
    AT_ERRCOUNT  err_count = 0;
    CHAR         file_name[PATH_MAX];
    const int    kExtraTextSize = 1024;
    CHAR         extra_text[kExtraTextSize];
    INT          line_number = 0;
    AT_ERRCODE   err_code = 0;
    UINT         level = 0;
    AT_INT       value1 = 0;
    AT_INT       value2 = 0;

    err_count = IG_err_count_get();

    int count=0;
    for (int i = 0; i < err_count && count < err_count; ++i, ++count) {
        IG_err_record_get(i,
            file_name,
            sizeof(file_name),
            &line_number,
            &err_code,
            &level,
            &value1,
            &value2,
            extra_text,
            kExtraTextSize);

        LPSTR err_description = NULL;
        for (AT_ERRCOUNT find = 0; ; ++find) {
            if (ErrString[find].ErrCode == err_code) {
                err_description = ErrString[find].ErrString;
                break;
            } else if (ErrString[find].ErrCode == IGE_LAST_ERROR_NUMBER) {
                err_description = error_not_found;
                break;
            }
        }

        printf("\n%s(%d) Error[code:%d, level:%d, val1:%d, val2:%d] (%s) - %s\n", file_name, line_number, err_code, level, value1, value2, err_description, extra_text);
    }
    exit(2);
}

void analyzeImage(char* data) {
    printf("Analyzing image data\n");

    BX_AnalyzeParameters  params = BX_DefaultAnalyzeParameters;
    BX_AnalyzeResult      result;
    BX_Status             status;

    params.Orientation = BX_Orientation_HorizontalVerticalDiagonal;
    params.BarcodeTypes = BX_BarcodeType_All;
    status = BX_analyze_dib(&params, data, &result);
    if( BX_Status_OK != status ) {
        printf("ERRORS:\n");
        for( int i=0; i<result.ErrorCount; ++i ) {
            BX_Error *error = &(result.BarcodeErrors[i]);
            printf("\t%d: %s\n\t%s\n", error->ErrorStatus, error->ErrorID, error->ErrorMessage);
        }
        return;
    }

    if( 0 == result.BarcodeResultsLength ) {
        printf("No results\n");
        BX_free_analyze_result( &result );
        return;
    }

    for( int i=0; i<result.BarcodeResultsLength; ++i ) {
        BX_BarcodeResult *res = &result.BarcodeResults[i];
        printf("\tResult #%d\n", i+1);
        printf("\t\tBarcode Type: %s\n", res->Name);
        printf("\t\tValue: %s\n", res->Value);
        printf("\t\tConfidence: %ld\n", res->Confidence);
    }

    BX_free_analyze_result( &result );
}

void readFile(const char *fileName) {
    const size_t kBufferLength = PATH_MAX;
    char   inputFilenameBuffer[kBufferLength]  = {0};
    strncpy(inputFilenameBuffer, fileName, strlen(fileName));

    // Check if file is present
    struct stat buffer;
    if (stat(inputFilenameBuffer, &buffer) != 0) {
        printf("File \"%s\" not found!\n", inputFilenameBuffer);
        exit(1);
    }
    printf("Reading image %s\n", fileName);

    // Open the input file in IG
    HIGEAR hIGear;
    AT_UINT fileSize = 0;
    AT_ERRCOUNT nErrCount = IG_load_file(inputFilenameBuffer, &hIGear);
    if (nErrCount > 0) {
        ImageGearErrOut();
        exit(1);
    }

    // Calculate the footprint of the BMP export
    nErrCount = IG_fltr_save_mem_size_calc(hIGear, 0, 0, IG_SAVE_BMP_UNCOMP, 1, 0, &fileSize);
    if (nErrCount > 0) {
        ImageGearErrOut();
        exit(1);
    }

    // Export the BMP
    char* dataBuffer = (char*)malloc(fileSize);
    nErrCount = IG_fltr_save_mem(hIGear, dataBuffer, 0, fileSize, IG_SAVE_BMP_UNCOMP, 1, true, &fileSize);
    if (nErrCount > 0) {
        ImageGearErrOut();
        exit(1);
    }

    // Offset 14 bytes for BMP file header
    analyzeImage( dataBuffer + 14 );

    free(dataBuffer);
    IG_image_delete(hIGear);
}

int main(const int argc, const char **argv) {
    if( argc < 2 ) {
        printf("Usage: readbarcodesig [filename [filename [...]]]\n");
        return -1;
    }

    setupLicense( );
    setupImageGear( );
    for(int i=1; i<argc; ++i) {
        readFile( argv[i] );
        printf("\n");
    }

    return 0;
}
