#pragma once
//////////////////////////////////////////////////////////////////////////
// Copyright� 2015-2023, Accusoft Corporation. All rights reserved.     //
//////////////////////////////////////////////////////////////////////////

#ifndef __BARCODEXPRESS_H__
#define __BARCODEXPRESS_H__

#define BX_MAX_ERROR_COUNT 64

#ifdef _WINDOWS
#include <Windows.h>
#endif


#include <stdbool.h>
#include <stdint.h>


#ifdef __cplusplus
extern "C" {
#endif

/////////////////////////////////////////////
// Status codes returned by BarcodeXpress
/////////////////////////////////////////////
/// <summary>
/// Specifies the possible states of the barcode engine.
/// </summary>
/// <remarks>
/// Any function call to the engine will return a status. The status will indicate if
/// the operation was successful.
/// </remarks>
typedef enum
{
    BX_Status_OK = 0,
    BX_Status_OutOfMemory = 9000,
    BX_Status_InvalidParameter = 9001,
    BX_Status_ParameterOutOfRange = 9002,
    BX_Status_InvalidBitmap = 9003,
    BX_Status_UnableToReadBitmap = 9004,
    BX_Status_UnsupportedBarcodeType = 9005,
    BX_Status_InvalidIndex = 9006,
    BX_Status_InvalidBarcodeType = 9007,
    BX_Status_ImageWidthTooSmall = 9008,
    BX_Status_ImageHeightTooSmall = 9009,
    BX_Status_UnableCreateUPCE = 9010,
    BX_Status_InvalidLicense = 9011,
    BX_Status_Timeout = 9012,
    BX_Status_InternalError = 9014,
    BX_Status_InvalidArea = 9015,

    BX_Status_UnsupportedBitDepth = 9022,
    BX_Status_NullResults = 9023,
    BX_Status_InvalidDIB = 9024,

    BX_Status_PDF417EncodingError = 9100,
    BX_Status_InvalidRowCount = 9101,
    BX_Status_InvalidColumnCount = 9102,
    BX_Status_SymbolCountError = 9103,
    BX_Status_ValueTooLarge = 9104,
    BX_Status_InvalidErrorCorrection = 9105,
    BX_Status_ErrorCorrectionLevelTooHigh = 9106,

    BX_Status_UnableToReadParameters = 9300,
    BX_Status_UnableToReadDIB = 9301,
    BX_Status_InvalidResults = 9302,
    BX_Status_UnableToCreateResults = 9303,
    BX_Status_NoLicenseFound = 9304,
    BX_Status_EncodeError = 9305
} BX_Status;

/////////////////////////////////////////////
// Types of barcodes
/////////////////////////////////////////////
/// <summary>
/// Specifies the types of barcodes that can be searched for or created.
/// </summary>
/// <remarks>
/// Any single barcode type or a combination of barcode types can be searched for.
/// To search for multiple barcode types OR together the values.
/// If you specify multiple barcodes types and one of them is BX_BarcodeType_Unknown
/// then only the explicity specified types will be searched for, and not all the other
/// types included in BX_BarcodeType_Unknown. The same applies to BX_BarcodeType_PatchCode. 
/// If you search for BX_BarcodeType_PatchCode and any other type, the other type will
/// be searched for. BX_BarcodeType_PatchCode must be the only specified type
/// if you wish to search for them."
/// </remarks>

typedef uint64_t BX_BarcodeType;

#define BX_BarcodeType_Unknown              0ULL
#define BX_BarcodeType_Industrial2of5       (1ULL << 0)
#define BX_BarcodeType_Interleaved2of5      (1ULL << 1)
#define BX_BarcodeType_IATA2OF5             (1ULL << 2)
#define BX_BarcodeType_Datalogic2of5        (1ULL << 3)
#define BX_BarcodeType_Inverted2of5         (1ULL << 4)
#define BX_BarcodeType_BCDMatrix            (1ULL << 5)
#define BX_BarcodeType_Matrix2of5           (1ULL << 6)
#define BX_BarcodeType_Code32               (1ULL << 7)
#define BX_BarcodeType_Code39               (1ULL << 8)
#define BX_BarcodeType_Codabar              (1ULL << 9)
#define BX_BarcodeType_Code93               (1ULL << 10)
#define BX_BarcodeType_Code128              (1ULL << 11)
#define BX_BarcodeType_EAN13                (1ULL << 12)
#define BX_BarcodeType_EAN8                 (1ULL << 13)
#define BX_BarcodeType_UPCA                 (1ULL << 14)
#define BX_BarcodeType_UPCE                 (1ULL << 15)
#define BX_BarcodeType_Add5                 (1ULL << 16)
#define BX_BarcodeType_Add2                 (1ULL << 17)
#define BX_BarcodeType_EAN128               (1ULL << 18)
#define BX_BarcodeType_PatchCode            (1ULL << 19)
#define BX_BarcodeType_PostNet              (1ULL << 20)
#define BX_BarcodeType_PDF417               (1ULL << 21)
#define BX_BarcodeType_DataMatrix           (1ULL << 22)
#define BX_BarcodeType_Code39Ext            (1ULL << 23)
#define BX_BarcodeType_Code93Ext            (1ULL << 24)
#define BX_BarcodeType_QRCode               (1ULL << 25)
#define BX_BarcodeType_IntelligentMail      (1ULL << 26)
#define BX_BarcodeType_RoyalPost4State      (1ULL << 27)
#define BX_BarcodeType_AustralianPost4State (1ULL << 28)
#define BX_BarcodeType_Aztec                (1ULL << 29)
#define BX_BarcodeType_GS1Databar           (1ULL << 30)
#define BX_BarcodeType_UPU4State            (1ULL << 31)
#define BX_BarcodeType_MicroPDF417          (1ULL << 32)
#define BX_BarcodeType_Planet               (1ULL << 33)
#define BX_BarcodeType_MicroQRCode          (1ULL << 34)

#define BX_BarcodeType_All1D  (BX_BarcodeType_Add2 | \
                               BX_BarcodeType_Add5 | \
                               BX_BarcodeType_AustralianPost4State | \
                               BX_BarcodeType_BCDMatrix | \
                               BX_BarcodeType_Codabar | \
                               BX_BarcodeType_Code32 | \
                               BX_BarcodeType_Code39 | \
                               BX_BarcodeType_Code39Ext | \
                               BX_BarcodeType_Code93 | \
                               BX_BarcodeType_Code93Ext | \
                               BX_BarcodeType_Code128 | \
                               BX_BarcodeType_Datalogic2of5 | \
                               BX_BarcodeType_EAN8 | \
                               BX_BarcodeType_EAN13 | \
                               BX_BarcodeType_EAN128 | \
                               BX_BarcodeType_GS1Databar | \
                               BX_BarcodeType_IATA2OF5 | \
                               BX_BarcodeType_Industrial2of5 | \
                               BX_BarcodeType_IntelligentMail | \
                               BX_BarcodeType_Interleaved2of5 | \
                               BX_BarcodeType_Inverted2of5 | \
                               BX_BarcodeType_Matrix2of5 | \
                               BX_BarcodeType_PatchCode | \
                               BX_BarcodeType_Planet | \
                               BX_BarcodeType_PostNet | \
                               BX_BarcodeType_RoyalPost4State | \
                               BX_BarcodeType_UPCA | \
                               BX_BarcodeType_UPCE | \
                               BX_BarcodeType_UPU4State)

#define BX_BarcodeType_All2D  (BX_BarcodeType_Aztec | \
                               BX_BarcodeType_DataMatrix | \
                               BX_BarcodeType_PDF417 | \
                               BX_BarcodeType_MicroPDF417 | \
                               BX_BarcodeType_QRCode | \
                               BX_BarcodeType_MicroQRCode)

#define BX_BarcodeType_All  (BX_BarcodeType_All1D | BX_BarcodeType_All2D)


/////////////////////////////////////////////
// Structures associated with Read calls
/////////////////////////////////////////////
/// <summary>
/// Specifies the module color of a barcode.
/// </summary>
/// <remarks>
/// <para>
/// Set this value appropriately to improve recognition accuracy and speed.
/// </para>
typedef enum
{
    /// <summary>
    /// Searches for barcodes with either black or white module ink color.
    /// </summary>
    BX_InkColor_Unknown = 0,
    /// <summary>
    /// Searches for barcodes with a black ink color (i.e. bars are black and spaces are white).
    /// </summary>
    BX_InkColor_Black = 1,
    /// <summary>
    /// Searches for barcodes with a white ink color (i.e. bars are white and spaces are black).
    /// </summary>
    BX_InkColor_White = 2
} BX_InkColor;

/// <summary>
/// Specifies the direction in which to search for barcodes.
/// </summary>
/// <remarks>
/// If the direction of a barcode is known then setting this value appropriately can improve performance.
/// </remarks>
typedef enum
{
    BX_Orientation_HorizontalVertical = 0,
    BX_Orientation_Horizontal = 1,
    BX_Orientation_Vertical = 2,
    BX_Orientation_HorizontalVerticalDiagonal = 3
} BX_Orientation;

/// <summary>
/// Specifies the method of decoding the custom section for Australian Post 4 State barcode
/// which supports a customer section in both the Customer Barcode 2 and Customer Barcode 3 formats.
/// </summary>
/// <remarks>
/// <para>The recommended bar encoding in this section is either Table-N or Table-C
/// encoding, but the customer is also free to put their own encoding in this section.</para>
/// <para>The type of encoding, in the customer section, is not stored in the barcode
/// format. The end customer must know the encoding prior to decoding the barcode.</para>
/// </remarks>
typedef enum
{
    /// <summary>
    /// Returns the bar states of the customer section. This will be a string of A D F T values where:
    /// A = Ascending Bar.
    /// D = Descending Bar.
    /// F = Full Bar.
    /// T = Tracking Bar.
    /// </summary>
    BX_AustralianPostDecodeType_BarStates = 0,
    /// <summary>
    /// The bars in the customer section are decoded using the "N" Table defined by the Australian Post 4 State Barcode specification.
    /// This will result in numeric values for the custom section of the barcode value.
    /// </summary>
    BX_AustralianPostDecodeType_TableN = 1,
    /// <summary>
    /// The bars in the customer section are decoded using the "C" Table defined by the Australian Post 4 State Barcode specification.
    /// This will result in Alphanumeric values for the custom section of the barcode value.
    /// </summary>
    BX_AustralianPostDecodeType_TableC = 2,
    /// <summary>
    /// Bars in the customer section will not be decoded.
    /// </summary>
    BX_AustralianPostDecodeType_NoCustomDecode = 3
} BX_AustralianPostDecodeType;

/// <summary>
/// Specifies the types of possible state transitions in the barcode reader's internal state.
/// </summary>
/// <remarks>
/// <para>
/// Many barcode types are encoded as a state machine. When reading the data encoded therein,
/// the machine will often move from one state to another. This information give decoders more information as to the inner context of the barcode's information.</para>
/// </remarks>
typedef enum
{
    BX_ModeTransitionType_QR_None = 0,
    BX_ModeTransitionType_QR_Numeric,
    BX_ModeTransitionType_QR_Alpha,
    BX_ModeTransitionType_QR_Byte,
    BX_ModeTransitionType_QR_Kanji,
    BX_ModeTransitionType_QR_ECI,
    BX_ModeTransitionType_QR_FNC1First,
    BX_ModeTransitionType_QR_FNC1Second,
    BX_ModeTransitionType_QR_StructuredAppend
} BX_ModeTransitionType;

/// <summary>
/// Stores information relating to any state changes that occur inside the barcode decoding process..
/// </summary>
/// <remarks>
/// Note: Barcode Xpress currently supports listing the state transitions for QR Code barcodes.
/// For other barcode types, there will be no mode transitions.
/// </remarks>
typedef struct tagBX_ModeTransition
{
    /// <summary>
    /// The index in the barcode Value where a transitions occurs.
    /// </summary>
    long Index;

    /// <summary>
    /// Specifies the mode of the new section starting at the index.
    /// </summary>
    BX_ModeTransitionType ModeType;
} BX_ModeTransition;

/// <summary>
/// Represents a point in the image.
/// </summary>
/// <remarks>
/// Note: distances are calculated from the top left hand corner of the image.
/// </remarks>
typedef struct tagBX_Point
{
    long X; /// The horizontal distance from the left edge of the image.
    long Y; /// The vertical distance from the top edge of the image.
} BX_Point;

/// <summary>
/// Represents a rectangular region on the image.
/// </summary>
/// <remarks>
/// Note: distances are calculated from the top left hand corner of the image.
/// </remarks>
typedef struct tagBX_Rectangle
{
    long Left; // The horizontal distance of the left edge of the area from the left edge of the image.
    long Top; // The vertical distance of the top edge of the area from the top edge of the image.
    long Right; // The horizontal distance of the right edge of the area from the left edge of the image.
    long Bottom; // The vertical distance of the bottom edge of the area from the top edge of the image.
} BX_Rectangle;

/// <summary>
/// Specifies the parameters that the barcode engine will use when searching an image for barcodes.
/// </summary>
typedef struct tagBX_AnalyzeParameters
{
    /// <summary>
    /// Specifies the version of the analyze parameters structure.
    /// The only valid value is 1. Do not change this to any other value.
    /// </summary>
    long ParametersVersion;

    /// <summary>
    /// Specifies if a checksum value is appended to the barcode during recognition.
    /// </summary>
    /// <remarks>
    /// If this property is set to true, checksum values will be appended to the result for barcode types that require a checksum.
    ///	This property does not have an effect for barcode types which do not have a checksum or for which the checksum is optional.
    /// <para>
    ///	<strong>Note:</strong> The UPC and EAN codes always append the checksum character to the barcode results.
    /// This is done regardless of this property's value.
    /// </para>
    /// <para>The <strong>Default value</strong> is <strong>false</strong></para>
    /// </remarks>
   bool AppendCheckSum;

    /// <summary>
    /// Specifies the area of an image to analyze for barcode recognition.
    /// </summary>
    /// <remarks>
    /// To search the entire image, all of the values must be set to 0.
    /// If any value is set, then they must all be set.
    /// <para>The <strong>Default value</strong> is <strong>{0,0,0,0}</strong></para>
    /// </remarks>
    BX_Rectangle Area;

    /// <summary>
    ///	Specifies the types of barcodes to search during the recognition process.
    /// </summary>
    /// <remarks>
    /// <para>
    /// By default, the barcode engine searches for all 1D barcodes. The 1D barcode set
    /// includes all types, except <strong>PatchCode</strong>, <strong>PostNet</strong>, <strong>Planet</strong>,
    /// <strong>GS1 DataBar</strong>, <strong>IntelligentMail</strong>, <strong>Royal Post Mail 4-State</strong>,
    /// <strong>Australian Post 4-State</strong>, <strong>QR Code</strong>,
    /// <strong>PDF417</strong>, <strong>Aztec</strong> and <strong>DataMatrix</strong>.</para>
    /// <para>
    ///	Multiple types of barcodes can by searched for by setting this to the ORed value of the types desired.
    /// For example, to search for PDF417 and Code39 set the BarcodeTypes to BX_BARCODE_TYPE_CODE39 | BX_BARCODE_TYPE_PDF417.
	/// However, if you want to search for <strong>Patchcodes</strong>, you must only specify the PatchCodeBarcode type.
	/// If any other type is also selected, Barcode Xpress will search for that other type instead.
    /// </para>
    /// <para>
    /// The <strong>default value</strong> is <strong>BX_BarcodeType_Unknown</strong>.
    /// </para>
    /// </remarks>
    BX_BarcodeType BarcodeTypes;

    /// <summary>
    ///	Specifies if control characters are added to the barcode value.
    /// </summary>
    /// <remarks>
    /// <para>
    /// If this property is set to <strong>true</strong>, the barcode control characters are
    /// included in the barcode read value. This has no effect for barcodes which do not support
    /// control characters.
    /// </para>
    /// </remarks>
    bool IncludeControlCharacters;

    /// <summary>
    /// Specifies what color of the barcode to search for.
    /// </summary>
    /// <remarks>
    /// The <strong>default value</strong> is <strong>BX_InkColor_Black </strong>.
    /// </remarks>
    BX_InkColor InkColor;

    /// <summary>
    /// Specifies the maximum number of barcodes to return.
    /// </summary>
    /// <remarks>
    /// <para>
    /// If more than this number of barcodes is found in the image, then the barcode engine
    /// will on return the number of barcodes specified.
    /// </para>
    /// <para>
    /// The <strong>minimum value</strong> is <strong>1 </strong>.
    /// </para>
    /// <para>
    /// The <strong>default value</strong> is <strong>100</strong>.
    /// </para>
    /// </remarks>
    long MaximumBarcodes;

    /// <summary>
    /// Specifies the minimum size of a barcode to return.
    /// </summary>
    /// <para>
    ///	Setting this value to 0 indicates that any size barcode should be returned.
    /// Any of the 4 edges of a barcode may pass this minimum size test to be returned (eg. A MinimumBarcodeSize of 100 would return a barcode with edge-lengths of [80, 85, 90, 100] because the fourth edge length is greater-than-or-equal to the minimum size).
    /// </para>
    /// <para>
    /// The <strong>default value</strong> is <strong>0</strong>.
    /// </para>
    long MinimumBarcodeSize;

    /// <summary>
    /// Specifies which direction to search for a barcode on the image.
    /// </summary>
    /// <remarks>
    /// <para>
    /// Set this value to expand or limit which direction to search for a barcode.
    /// For example, to only find barcodes that are oriented vertically set this to BX_Orientation_Vertical.
    /// Any barcodes that are oriented horizontally or diagonally will not be returned.
    /// </para>
    /// The <strong>default value</strong> is <strong>BX_Orientation_HorizontalVertical</strong>.</para>
    /// </remarks>
    BX_Orientation Orientation;

    /// <summary>
    /// Specifies if the barcode engine should return possible barcodes.
    /// </summary>
    /// <remarks>
    /// <para>
    /// When this property is set to True, the barcode engine returns candidate barcodes
    /// which could not be decoded in the results list.
    /// </para>
    /// The <strong>default value</strong> is <strong>false</strong>.
    /// </remarks>
    bool ReturnPossibleBarcodes;

    /// <summary>
    /// Specifies the method of decoding the custom section for Australian Post 4 State barcode
    /// which supports a custom section in both the Customer Barcode 2 and Customer Barcode 3 formats.
    /// </summary>
    /// <remarks>
    /// The <strong>default value</strong> is <strong>BX_AustralianPostDecodeType_NoCustomDecode</strong>.
    /// </remarks>
    BX_AustralianPostDecodeType AustralianPostDecodeType;

    /// <summary>
    /// Specifies whether or not to decode Royal Mail barcodes of any length. 
    /// If set to false, only Royal Mail barcodes matches the lengths in certain specifications will be decoded.
    /// </summary>
    /// <remarks>
    /// The <strong>default value</strong> is <strong>false</strong>.
    /// </remarks>
    bool RoyalMailVariableLengthDecoding;

    /// <summary>
    /// When property is set to <strong>true</strong> then only regular (barcode has both start/stop patterns) barcodes are recognized.
    /// When property is set to <strong>false</strong> then both regular and barcodes without start/stop patterns are recognized.
    /// This is relevant for Code39 and Code39 Extended barcode symbologies.
    /// </summary>
    /// <remarks>
    /// The <strong>default value</strong> is <strong>true</strong>.
    /// </remarks>
    bool Code39StartStopPatternsAreMandatory;

    /// <summary>
    /// Specifies the scan distance in pixels between line sweeps when searching for 1D barcodes.
    /// </summary>
    /// <remarks>
    /// Valid values are 1 up to and including 10.
    /// The <strong>default value</strong> is <strong>5</strong>.
    /// </remarks>
    long ScanDistance;

    /// <summary>
    /// Enables the direct analysis of grayscale image processing.
    /// </summary>
    /// <remarks>
    /// When this value is set to True, the barcode engine will process a grayscale image along with processing a binarized version of the image.
    /// The <strong>default value</strong> is <strong>false</strong>.
    /// </remarks>
    bool GrayscaleProcessing;

    /// <summary>
    /// Enables additional scaled (reduced) image processing.
    /// </summary>
    /// <remarks>
    /// When this value is set to True, the barcode engine will reduce the original image and perform additional processing pass.
    /// The <strong>default value</strong> is <strong>false</strong>.
    /// </remarks>
    bool AdditionalReadingPass;

    /// <summary>
    /// This parameter should be set to null (0)
    /// </summary>
    /// <remarks>
    /// This parameter should be set to null (0)
    /// </remarks>
    void *DebugLogger;
} BX_AnalyzeParameters;


/// <summary>
/// Contains the 2D result information for a single detected barcode..
/// </summary>
typedef struct tagBX_BarcodeResult2D
{
    /// <summary>
    /// Specifies the number of rows detected in the barcode image.
    /// </summary>
    long RowsDetected;

    /// <summary>
    /// Specifies the number of columns detected in the barcode image.
    /// </summary>
    long ColumnsDetected;

    /// <summary>
    /// Specifies the number of rows in the barcode as defined by the indicator pattern.
    /// </summary>
    long Rows;

    /// <summary>
    /// Specifies the number of columns in the barcode as defined by the indicator pattern.
    /// </summary>
    /// <remarks>
    ///	<para>
    /// Column Count for PDF417 barcodes is the number of codeword columns.
    /// </para>
    /// </remarks>
    long Columns;

    /// <summary>
    /// Specifies the amount of error correction detected for barcodes that support it.
    /// </summary>
    /// <remarks>
    ///	<para>
    /// Error correction is valid for QR barcodes and PDF417 barcodes.
    /// </para>
    /// <para>
    /// The error correction level for the QRCode ranges from a low of -1 to a high of 3.  
    /// </para>
    /// <para>
    /// The error correction level for the PDF417 ranges from a low of -1 (2 code words of error correction)
    /// to a high of 8 (512 code words of error correction).  
    /// </para>
    /// </remarks>
    long ErrorCorrectionLevel;
} BX_BarcodeResult2D;

/// <summary>
/// Contains the result of a single detected barcode.
/// </summary>
typedef struct tagBX_BarcodeResult
{
    /// <summary>
    /// Specifies a rectangular area on the image containing the barcode.
    /// </summary>
    /// <remarks>
    ///	This area does not necessarily correspond to the corner points of a barcode.
    /// </remarks>
    BX_Rectangle Area;

    /// <summary>
    /// Specifies the upper-left corner point of the barcode.
    /// </summary>
    BX_Point Point1;

    /// <summary>
    /// Specifies the upper-right corner point of the barcode.
    /// </summary>
    BX_Point Point2;

    /// <summary>
    /// Specifies the lower-right corner point of the barcode.
    /// </summary>
    BX_Point Point3;

    /// <summary>
    /// Specifies the lower-left corner point of the barcode.
    /// </summary>
    BX_Point Point4;

    /// <summary>
    /// Specifies the confidence value that the barcode result is accurate.
    /// </summary>
    /// <remarks>
    ///	Confidence values of 30 indicate that the barcode value could not be determined accurately.
    /// </remarks>
    long Confidence;

    /// <summary>
    /// Specifies the name of the detected barcode.
    /// </summary>
    /// <remarks>
    ///	This is a string representation of the barcode-type.
    /// </remarks>
    char* Name;

    /// <summary>
    /// Specifies the length of the barcode name string.
    /// </summary>
    long NameLength;

    /// <summary>
    /// Specifies the angle of skew for the recognized barcode.
    /// </summary>
    /// <remarks>
    ///	The skew angle is measured in degrees, with 0 degrees being horizontal, from the
    /// horizontal axis, to the top edge of the barcode.
    /// </remarks>
    long Skew;

    /// <summary>
    /// Specifies the type of barcode that is detected.
    /// </summary>
    /// <remarks>
    ///	Note: If a barcode is not decoded and ReturnPossibleBarcodes is true, then the BarcodeType returned is BX_BARCODE_TYPE_UNKNOWN.
    /// </remarks>
    BX_BarcodeType BarcodeType;

    /// <summary>
    /// Specifies if the checksum for the recognized barcode is valid.
    /// </summary>
    /// <remarks>
    ///	<para>
    /// For barcodes with a checksum, it returns whether the checksum is valid. A value of 1 represents True,
    /// a value of 0 represents False.
    /// </para>
    /// <para>For barcodes without a checksum, it returns False.</para>
    /// <para>For barcodes with error correction, it returns True.</para>
    /// </remarks>
    bool CheckSumValid;

    /// <summary>
    /// Specifies the number of characters in the recognized checksum.
    /// </summary>
    long CheckSumLength;

    /// <summary>
    /// Specifies a string containing the value of a recognized barcode.
    /// </summary>
    char* Value;

    /// <summary>
    /// Specifies the length (or number of characters) in the recognized varcode value.
    /// </summary>
    long ValueLength;

    /// <summary>
    /// Specifies the Mode Transitions for barcodes that support it.
    /// </summary>
    /// <remarks>
    ///	<para>
    /// When reading a barcode, many symbologies will go through state changes that
    /// may give extra context about the content types inside the barcode (eg. Kanji).
    /// This gives the user extra information about the inner workings
    /// of the barcode's state.
    /// </para>
    /// <para><strong>Note:</strong> Currently only enabled with QR Codes.</para>
    /// </remarks>
    BX_ModeTransition* ModeTransitions;

    /// <summary>
    /// Specifies the number of transitions detected in the barcode.
    /// </summary>
    long ModeTransitionsLength;

    /// <summary>
    /// Specifies the 2D result properties
    /// </summary>
    BX_BarcodeResult2D Result2D;
} BX_BarcodeResult;

/// <summary>
/// Contains information for a single error that may be generated by the barcode engine.
/// </summary>
typedef struct tagBX_Error
{
    BX_Status ErrorStatus; // The state of the barcode engine when the error occurs.
    const char* ErrorID; // A string describing the error that has occurred.
    const char* ErrorMessage; // A string providing additional information about the error.
} BX_Error;

/// <summary>
/// Specifies the results of the analysis process.
/// </summary>
typedef struct tagBX_AnalyzeResult
{
    /// <summary>
    /// Points to an array containing the results of barcodes found during the analysis.
    /// </summary>
    /// <remarks>
    /// Before accessing this member, check that no error has occurred and that results were returned.
    /// </remarks>
    BX_BarcodeResult* BarcodeResults;

    /// <summary>
    /// Points to an array of errors returned by the barcode engine.
    /// </summary>
    /// <remarks>
    /// Any errors that occur during the analysis process are added to this array. Multiple errors may be returned by a single function call.
    /// </remarks>
   BX_Error BarcodeErrors[BX_MAX_ERROR_COUNT];

    /// <summary>
    /// Specifies the number of barcodes found in an image.
    /// </summary>
    /// <remarks>
    /// If no barcodes are found then this value will be 0.
    /// </remarks>
    long BarcodeResultsLength;

    /// <summary>
    /// Specifies the number of errors that occurred while searching for a barcode.
    /// </summary>
    /// <remarks>
    /// If no errors occurred this value will be 0.
    /// </remarks>
    long ErrorCount;
} BX_AnalyzeResult;

/////////////////////////////////////////////
// Structures associated with write calls
/////////////////////////////////////////////

/// <summary>
/// Specifies the possible sizes of a DataMatrix barcode.
/// </summary>
/// <remarks>
/// The rows and columns are encoded in the value of the enumerations using this equation:
/// RowsXColumnsY = X * 256 + Y. AutoRowsAndColumns is -1.
/// </remarks>
typedef enum tagBX_DataMatrixSize
{
    BX_DataMatrixSize_AutoRowsAndColumns = -1,
    BX_DataMatrixSize_Rows10Columns10 = (10*256 + 10),
    BX_DataMatrixSize_Rows12Columns12 = (12*256 + 12),
    BX_DataMatrixSize_Rows14Columns14 = (14*256 + 14),
    BX_DataMatrixSize_Rows16Columns16 = (16*256 + 16),
    BX_DataMatrixSize_Rows18Columns18 = (18*256 + 18),
    BX_DataMatrixSize_Rows20Columns20 = (20*256 + 20),
    BX_DataMatrixSize_Rows22Columns22 = (22*256 + 22),
    BX_DataMatrixSize_Rows24Columns24 = (24*256 + 24),
    BX_DataMatrixSize_Rows26Columns26 = (26*256 + 26),
    BX_DataMatrixSize_Rows32Columns32 = (32*256 + 32),
    BX_DataMatrixSize_Rows36Columns36 = (36*256 + 36),
    BX_DataMatrixSize_Rows40Columns40 = (40*256 + 40),
    BX_DataMatrixSize_Rows44Columns44 = (44*256 + 44),
    BX_DataMatrixSize_Rows48Columns48 = (48*256 + 48),
    BX_DataMatrixSize_Rows52Columns52 = (52*256 + 52),
    BX_DataMatrixSize_Rows64Columns64 = (64*256 + 64),
    BX_DataMatrixSize_Rows72Columns72 = (72*256 + 72),
    BX_DataMatrixSize_Rows80Columns80 = (80*256 + 80),
    BX_DataMatrixSize_Rows88Columns88 = (88*256 + 88),
    BX_DataMatrixSize_Rows96Columns96 = (96*256 + 96),
    BX_DataMatrixSize_Rows104Columns104 = (104*256 + 104),
    BX_DataMatrixSize_Rows120Columns120 = (120*256 + 120),
    BX_DataMatrixSize_Rows132Columns132 = (132*256 + 132),
    BX_DataMatrixSize_Rows144Columns144 = (144*256 + 144),
    BX_DataMatrixSize_Rows8Columns18 = (8*256 + 18),
    BX_DataMatrixSize_Rows8Columns32 = (8*256 + 32),
    BX_DataMatrixSize_Rows12Columns26 = (12*256 + 26),
    BX_DataMatrixSize_Rows12Columns36 = (12*256 + 36),
    BX_DataMatrixSize_Rows16Columns36 = (16*256 + 36),
    BX_DataMatrixSize_Rows16Columns48 = (16*256 + 48)
} BX_DataMatrixSize;

/// <summary>
/// Specifies the possible sizes of a QR barcode.
/// </summary>
typedef enum tagBX_QRCodeSize
{
    BX_QRCodeSize_AutoRowsAndColumns = -1,
    BX_QRCodeSize_Rows21Columns21 = 21,
    BX_QRCodeSize_Rows25Columns25 = 25,
    BX_QRCodeSize_Rows29Columns29 = 29,
    BX_QRCodeSize_Rows33Columns33 = 33,
    BX_QRCodeSize_Rows37Columns37 = 37,
    BX_QRCodeSize_Rows41Columns41 = 41,
    BX_QRCodeSize_Rows45Columns45 = 45,
    BX_QRCodeSize_Rows49Columns49 = 49,
    BX_QRCodeSize_Rows53Columns53 = 53,
    BX_QRCodeSize_Rows57Columns57 = 57,
    BX_QRCodeSize_Rows61Columns61 = 61,
    BX_QRCodeSize_Rows65Columns65 = 65,
    BX_QRCodeSize_Rows69Columns69 = 69,
    BX_QRCodeSize_Rows73Columns73 = 73,
    BX_QRCodeSize_Rows77Columns77 = 77,
    BX_QRCodeSize_Rows81Columns81 = 81,
    BX_QRCodeSize_Rows85Columns85 = 85,
    BX_QRCodeSize_Rows89Columns89 = 89,
    BX_QRCodeSize_Rows93Columns93 = 93,
    BX_QRCodeSize_Rows97Columns97 = 97,
    BX_QRCodeSize_Rows101Columns101 = 101,
    BX_QRCodeSize_Rows105Columns105 = 105,
    BX_QRCodeSize_Rows109Columns109 = 109,
    BX_QRCodeSize_Rows113Columns113 = 113,
    BX_QRCodeSize_Rows117Columns117 = 117,
    BX_QRCodeSize_Rows121Columns121 = 121,
    BX_QRCodeSize_Rows125Columns125 = 125,
    BX_QRCodeSize_Rows129Columns129 = 129,
    BX_QRCodeSize_Rows133Columns133 = 133,
    BX_QRCodeSize_Rows137Columns137 = 137,
    BX_QRCodeSize_Rows141Columns141 = 141,
    BX_QRCodeSize_Rows145Columns145 = 145,
    BX_QRCodeSize_Rows149Columns149 = 149,
    BX_QRCodeSize_Rows153Columns153 = 153,
    BX_QRCodeSize_Rows157Columns157 = 157,
    BX_QRCodeSize_Rows161Columns161 = 161,
    BX_QRCodeSize_Rows165Columns165 = 165,
    BX_QRCodeSize_Rows169Columns169 = 169,
    BX_QRCodeSize_Rows173Columns173 = 173,
    BX_QRCodeSize_Rows177Columns177 = 177
} BX_QRCodeSize;
/// <summary>
/// Returns the values used during the writing process.
/// </summary>
typedef struct tagBX_CreateResult2D
{
    /// <summary>
    /// The number of rows created. 
    /// <remarks>
    ///      If the BX_CreateParameters2D.Rows is -1, then this value will be automatically determined. 
    ///	     Otherwise, this value will be the same as that parameter.
    /// </summary>
    long RowsCreated; 
    /// <summary>
    /// The number of columns created. 
    /// <remarks>
    ///      If the BX_CreateParameters2D.Columns is -1, then this value will be automatically determined. 
    ///	     Otherwise, this value will be the same as that parameter.
    /// </summary>
    long ColumnsCreated;

}BX_CreateResult2D;
/// <summary>
/// Specifies the results of attempting to create a barcode.
/// </summary>
typedef struct tagBX_CreateResult
{
    /// <summary>
    /// An array of errors that occurred during the barcode creation process.
    /// </summary>
    BX_Error BarcodeErrors[BX_MAX_ERROR_COUNT];

    /// <summary>
    /// The number of errors that occurred.
    /// </summary>
    long ErrorCount;
    
    /// <summary>
    /// Results from creating a 2D barcode. 
    /// <remarks>
    /// The values in this array will only be valid if the error count is 0. 
    /// </remarks>
    /// </summary>
    BX_CreateResult2D Results2D; 
} BX_CreateResult;

/// <summary>
/// Specifies the valid values for showing text under the barcode.
/// </summary>
typedef enum
{
    BX_ShowTextInfo_None = 0, /// No text is displayed.
    BX_ShowTextInfo_Value = 1, /// The barcode value is displayed.
    BX_ShowTextInfo_ValueChecksum = 2, /// The barcode value and checksum are displayed.
    BX_ShowTextInfo_All = 3 /// All possible text is displayed.
} BX_ShowTextInfo;

/// <summary>
/// Specifies the parameters used to create a 1D barcode.
/// </summary>
typedef struct tagBX_CreateParameters1D
{
    /// <summary>
    /// Specifies whether to write a checksum for barcodes that optionally use checksums.
    /// </summary>
    /// <remarks>
    /// <para>
    /// When true, Barcode Xpress will calculate and write the checksum for barcode types
    /// that support an optional checksum digit(s).
    /// </para>
    /// <para>
    /// Currently, of the supported barcode types that Barcode Xpress can write, only types
    /// Industry 2 of 5, Interleaved 2 of 5, Code 39, and Code 39 Extended allows an optional
    /// checksum. This property is ignored for other barcode types that require checksums.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>false</strong>. </para>
    /// </remarks>
    bool CheckSum;

    /// <summary>
    /// Specifies the amount of gap between the outside text and the barcode when
    /// creating a barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// Outside text is only written with certain barcodes where this is standard.
    /// The barcode types that support a text notch are UPC-A, UPC-E, EAN-8 and EAN-13.
    /// For all other types, this value is ignored.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>5 pixels</strong></para>
    /// </remarks>
    long HorizontalTextGap;

    /// <summary>
    /// Specifies the vertical offset text.
    /// </summary>
    /// <remarks>
    /// <para>
    /// A value less than 0 will raise the outside value text above the standard value text.
    /// A value greater than 0 will lower the outside value text below the standard value text.
    /// </para>
    /// <para>
    /// Outside value text is only written for certain barcodes where this is standard.
    /// The barcode types that support outside text are UPC-A, UPC-E, EAN-8 and EAN-13. For all
    /// other types this value is ignored.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>0 pixels</strong></para>
    /// </remarks>
    long OutsideVerticalTextOffset;

    /// <summary>
    /// Specifies if the barcode value is written as text adjacent to the created barcode.
    /// </summary>
    /// <remarks>
    /// <para>The <strong>default value</strong> is <strong>BX_ShowTextInfo_All</strong></para>
    /// </remarks>
    BX_ShowTextInfo ShowText;

    /// <summary>
    /// Specifies the notch percent of the barcode for the text value.
    /// </summary>
    /// <remarks>
    /// <para>
    /// This property is used by supported barcodes UPC-A and EAN13. Otherwise this value is
    /// ignored.
    /// </para>
    /// <para>
    /// Notch percent is only used with certain barcodes where this is standard. The
    /// barcode types that support a text notch are UPC-A, UPC-E, EAN-8 and EAN-13.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>10 percent</strong>.</para>
    /// <para><strong>Valid values: </strong>0 - 50 percent</para>
    /// </remarks>
    long TextNotchPercent;

    /// <summary>
    /// Specfies the amount of gap between the text and the barcode when creating a barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// This property value can be positive or negative. Positive values move the
    /// value text further away from the bottom of the bars while negative values move
    /// the text closer to the bars, even overlapping.
    /// </para>
    /// <para>
    /// <strong>Note: </strong>In the case where the value text overlaps the barcode,
    /// this may prevent the barcode from being recognized.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>0 pixels</strong></para>	/// </remarks>
    long VerticalTextGap;

#if defined(_WINDOWS) || defined(USE_ACCU_WIN_DEFS)
#ifdef USE_ACCU_WIN_DEFS
typedef void* HFONT;
#endif
    /// <summary>
    /// Gets and sets the font of the barcode written as text adjacent to the barcode being created.
    /// </summary>
    /// <remarks>
    /// <para><strong>Default value:</strong> Arial, 16, Normal</para>
    /// </remarks>
    HFONT TextFont;
    /// <summary>
    /// Gets and sets the font used to display the UPC values that appear outside of
    /// the left and right guard bars.
    /// </summary>
    /// <remarks>
    /// <para><strong>Default value:</strong> Arial, 16, Normal</para>
    /// </remarks>
    HFONT OutsideTextFont;
#endif

} BX_CreateParameters1D;

/// <summary>
/// Specifies the parameters used to create a 2D barcode.
/// </summary>
typedef struct tagBX_CreateParameters2D
{
    /// <summary>
    /// Specifies the number of rows to create.
    /// </summary>
    /// <remarks>
    ///	Set this to -1 to let the writer automatically choose the number of rows.
    /// </remarks>
    long Rows;

    /// <summary>
    /// Specifies the number of columns to create.
    /// </summary>
    /// <remarks>
    ///	Set this to -1 to let the writer automatically choose the number of columns.
    /// </remarks>
    long Columns;

    /// <summary>
    /// Specifies the amount of error correction to use for barcodes that support it.
    /// </summary>
    /// <remarks>
    ///	<para>
    /// Error correction is valid for QR barcodes and PDF417 barcodes.
    /// </para>
    /// <para>
    /// The error correction level for the QRCode ranges from a low of -1 to a high of 3.  A high error correction level should
    /// be used in application where symbol damage is expected.
    /// </para>
    /// <para>
    /// The error correction level for the PDF417 ranges from a low of -1 (2 code words of error correction)
    /// to a high of 8 (512 code words of error correction).  A high error correction level should
    /// be used in application where symbol damage is expected.
    /// </para>
    /// <para>
    ///  A -1 value indicates that the barcode writer chooses the best error correction level based on the number of
    /// code words in the data stream.
    /// </para>
    /// </remarks>
    long ErrorCorrectionLevel;
} BX_CreateParameters2D;

/// <summary>
/// Specifies the parameters used to create a barcode.
/// </summary>
typedef struct tagBX_CreateParameters
{
    /// <summary>
    /// Specifies the version of the create parameters structure.
    /// The only valid value is 1. Do not change this to any other value.
    /// </summary>
    long ParametersVersion;

    /// <summary>
    /// Specifies the value of the barcode to create.
    /// </summary>
    /// <remarks>
    /// <para>The <strong>default value</strong> is <strong>an empty string ""</strong>.</para>
    /// </remarks>
    const char* Value;

    /// <summary>
    /// Specifies the length of the Value string.
    /// </summary>
    long ValueLength;

    /// <summary>
    /// Specifies the type of barcode to create.
    /// </summary>
    BX_BarcodeType BarcodeType;

    /// <summary>
    /// Specifies the width of the image, in pixels, containing the barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    ///	If AutoSize is true, then this value is ignored.
    /// </para>
    /// <para>The <strong>default value</strong>is <strong>600</strong>.</para>
    /// </remarks>
    long Width;

    /// <summary>
    /// Specifies the height of the image, in pixels, containing the barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    ///	If AutoSize is true, then this value is ignored.
    /// </para>
    /// <para>The <strong>default value</strong>is <strong>600</strong>.</para>
    /// </remarks>
    long Height;

     /// <summary>
    /// Specifies the minimum bar width when creating a barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// This value is used to set the bar width in 1D and PDF417 barcodes and the
    /// module size (black and white squares that make up the barcode) in DataMatrix barcodes.
    /// </para>
    /// <para>
    /// If AdjustBarWidth parameter is set to true then this value is ignored.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>3</strong>.</para>
    /// </remarks>
    long MinimumBarWidth;

    /// <summary>
    /// Specifies the minimum height of the bars when creating a barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// <strong>Note:</strong> When writing a PDF417 set this value between 3 and 5.
    /// </para>
    /// <para>
    /// This value is used to set the bar height in 1D barcodes. PDF417 uses this
    /// value as a multiplier of the minimum barcode width to set the module height of each data row.
    /// </para>
    /// <para>Note: This property does not affect DataMatrix barcodes.</para>
    /// <para>The <strong>default value</strong> is <strong>100. </strong></para>
    /// </remarks>
    long MinimumBarHeight;

    /// <summary>
    /// Specifies the amount of white space that is created on the sides of the barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// The white space surrounding the barcode is also known as the quiet zone. The Barcode Xpress
    /// component requires a quiet zone for accurate barcode detection.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>50</strong>.</para>
    /// </remarks>
    long HorizontalWhiteSpace;


    /// <summary>
    /// Specifies the amount of white space that is created above and below the barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// The white space surrounding the barcode is also known as the quiet zone. The Barcode Xpress
    /// component requires a quiet zone for accurate barcode detection.
    /// </para>
    /// <para>The <strong>default value</strong> is <strong>50</strong>.</para>
    /// </remarks>
    long VerticalWhiteSpace;

    /// <summary>
    /// Specifies if dynamic adjustment of the bar widths during barcode
    /// creation is allowed.
    /// </summary>
    /// <remarks>
    /// <para>The <strong>default value</strong> is <strong>true</strong>.</para>
    /// </remarks>
    bool AdjustBarWidth;

    /// <summary>
    /// Specifies if the barcode area is automatically sized during barcode
    /// creation.
    /// </summary>
    /// <remarks>
    /// <para>When this property is set to true, the component will calculate the required height and
    /// width of the output image during barcode creation.</para>
    /// <para>The image height and width are calculated using the barcode value, minimum bar width,
    /// horizontal gaps, vertical gaps and the size of the text font. If the calculated height or width is
    /// greater than the user input height and width, the image size will be increased to create the
    /// barcode.
    /// </para>
    /// <para>
    /// When this property is set to false, the component will use the <see cref="Height"/> and
    /// <see cref="Width"/> parameters.
    /// </para>	
    /// <para>The <strong>default value</strong> is <strong>true</strong>.</para>
    /// </remarks>
    bool AutoSize;

    /// <summary>
    /// Specifies the parameters for creating a 1D barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// This value is used when the BarcodeType is set to a 1D barcode.
    /// If creating a 2D barcode then this value is ignored.
    /// </para>	
    /// </remarks>
    BX_CreateParameters1D Params1D;

    /// <summary>
    /// Specifies the parameters for creating a 2D barcode.
    /// </summary>
    /// <remarks>
    /// <para>
    /// This value is used when the BarcodeType is set to a 2D barcode.
    /// If creating a 1D barcode then this value is ignored.
    /// </para>	
    /// </remarks>
    BX_CreateParameters2D Params2D;
} BX_CreateParameters;

/////////////////////////////////////////////
// Default Parameter values
/////////////////////////////////////////////

extern const BX_AnalyzeParameters BX_DefaultAnalyzeParameters;
extern const BX_CreateParameters BX_DefaultCreateParameters;

/////////////////////////////////////////////
// Licensing API Calls
/////////////////////////////////////////////

/// <summary>
/// Sets the Barcode Xpress control runtime license deployment solution name.
/// </summary>
/// <param name="solutionName">A string containing the solution name.</param>
/// <remarks>
/// This is the solution name required for all runtime licensing.
/// Once purchased, the solution name can be obtained from the customer portal.
/// </remarks>
BX_Status BX_set_solution_name(const char* solutionName);

/// <summary>
/// Sets the BarcodeXpress control runtime license deployment solution keys.
/// </summary>
/// <param name="key1">A long value of key 1</param>
/// <param name="key2">A long value of key 2</param>
/// <param name="key3">A long value of key 3</param>
/// <param name="key4">A long value of key 4</param>
/// <remarks>
/// This is the solution key required for all runtime licensing.
/// The four integers are provided at the time of purchase of the runtime licenses along with the solution name.
/// Once purchased, they can be obtained from the customer portal.
/// </remarks>
BX_Status BX_set_solution_key(const long key1, const long key2, const long key3, const long key4);

/// <summary>
/// Sets the BarcodeXpress control runtime license deployment key.
/// </summary>
/// <param name="oemLicenseKey">A string containing the OEM license key required for OEM licensing.</param>
/// <remarks>
/// The OEM key needs to be set only if purchased. If you have not purchased an
/// OEM key then do not call this function.
/// </remarks>
BX_Status BX_set_oem_license_key(const char* oemLicenseKey);

/////////////////////////////////////////////
// Reader API Calls
/////////////////////////////////////////////

/// <summary>
/// Searches for barcodes in a memory image.
/// </summary>
/// <remarks>
/// <param name="params">A pointer to the BX_AnalyzeParameters structure specifying the options for analyzing an image.</param>
/// <param name="pPixels">A pointer to a block of memory containing the image to analyze.</param>
/// <param name="result">A pointer to the BX_AnalyzeResult structure. The results of the barcodes found in the image will be placed into this structure.</param>
/// </remarks>

BX_Status BX_analyze_memory(const BX_AnalyzeParameters* params, const long height, const long width, const long stride, const long bitsPerPixel,
    const void* pPixels, const unsigned int paletteColors, const void* palette, BX_AnalyzeResult* result);

/// <summary>
/// Searches for barcodes in an image.
/// </summary>
/// <remarks>
/// <param name="params">A pointer to the BX_AnalyzeParameters structure specifying the options for analyzing an image.</param>
/// <param name="dib">A pointer to a bitmap containing the image to analyze.</param>
/// <param name="result">A pointer to the BX_AnalyzeResult structure. The results of the barcodes found in the image will be placed into this structure.</param>
/// </remarks>
BX_Status BX_analyze_dib(const BX_AnalyzeParameters* params, const void* dib, BX_AnalyzeResult* result);

/// <summary>
/// Searches for barcodes in a bitmap file.
/// </summary>
/// <remarks>
/// <param name="params">A pointer to the BX_AnalyzeParameters structure specifying the options for recognizing barcodes in an image.</param>
/// <param name="filePath">A string specifying the file to analyze.</param>
/// <param name="result">A pointer to the BX_AnalyzeResult structure. The results of the barcodes found in the image will be placed into this structure.</param>
/// </remarks>
BX_Status BX_analyze_file(const BX_AnalyzeParameters* params, const char* filePath, BX_AnalyzeResult* result);

/// <summary>
/// Searches for barcodes in a bitmap encoded image in memory.
/// </summary>
/// <remarks>
/// <param name="params">A pointer to the BX_AnalyzeParameters structure specifying the options for recognizing barcodes in an image.</param>
/// <param name="image">A pointer to a bitmap containing the image to analyze.</param>
/// <param name="result">A pointer to the BX_AnalyzeResult structure. The results of the barcodes found in the image will be placed into this structure.</param>
/// </remarks>
BX_Status BX_analyze_bmp(const BX_AnalyzeParameters* params, void* bmp, unsigned long length, BX_AnalyzeResult* result);

/// <summary>
/// Releases the memory that has been allocated by a previous call to one of the analyze functions.
/// </summary>
/// <remarks>
/// <param name="result">A pointer to the BX_AnalyzeResult structure to release. </param>
/// </remarks>
BX_Status BX_free_analyze_result(BX_AnalyzeResult* result);

/////////////////////////////////////////////
// Writer API Calls
/////////////////////////////////////////////

/// <summary>
/// Creates a barcode in an image and returns a pointer to it.
/// </summary>
/// <remarks>
/// <param name="params">A pointer to the BX_CreateParameters structure specifying the options to use for creating the barcode image.</param>
/// <param name="dib">A pointer to a bitmap that is created during the creation process.</param>
/// <param name="result">A pointer to the BX_CreateResult structure.</param>
/// </remarks>
BX_Status BX_create_dib(const BX_CreateParameters* params, void** dib, BX_CreateResult* result);

/// <summary>
/// Creates a barcode in an image and writes that to a file.
/// </summary>
/// <remarks>
/// <param name="params">A pointer to the BX_CreateParameters structure specifying the options to use for creating the barcode image.</param>
/// <param name="filePath">A string specifying the filename of the output image.</param>
/// <param name="result">A pointer to the BX_CreateResult structure.</param>
/// </remarks>
BX_Status BX_create_file(const BX_CreateParameters* params, const char* filePath, BX_CreateResult* result);

/// <summary>
/// Releases the memory that has been allocated by a previous call to the create dib function.
/// </summary>
/// <remarks>
/// <param name="dib">A pointer to a bitmap structure to release. </param>
/// </remarks>
BX_Status BX_free_dib(void** dib);

#ifdef __cplusplus
}  /* End of the extern "C" definition */
#endif

#endif /* __BARCODEXPRESS_H__ */
