import com.accusoft.barcodexpress.*;
import javax.imageio.ImageIO;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

public class ReadBarcodes {
    public static void main(String[] args) {
        BarcodeXpress barcodeXpress = new BarcodeXpress();

        // The SetSolutionName, SetSolutionKey and possibly the SetOEMLicenseKey method must be
        // called to distribute the runtime.  Note that the SolutionName, SolutionKey and
        // OEMLicenseKey values shown below are only examples.
        // BarcodeXpress.setSolutionName("YourSolutionName");
        // BarcodeXpress.setSolutionKey(1234,1234,1234,1234);
        // BarcodeXpress.setOemLicenseKey("2.0.YourOEMLicenseKeyGoesHere");

        BarcodeReader reader = barcodeXpress.getReader();
        String[] fileNames = parseCommandLineArgs(args, reader);
        for (int i = 0; i < fileNames.length; i++) {
            readerAnalyze(reader, fileNames[i]);
        }
    }

    public static void readerAnalyze(BarcodeReader reader, String inputFilePath) {
        System.out.println("Analyzing file " + inputFilePath);
        System.out.println();

        try {
            File inputFile = new File(inputFilePath);
            BufferedImage bufferedImage = ImageIO.read(inputFile);

            Result[] results = reader.analyze(bufferedImage);
            System.out.println(results.length + " barcodes found");

            for (int i = 0; i < results.length; i++) {
                System.out.println("#" + (i+1));
                System.out.println("Value: " + results[i].getValue());
                System.out.println("Type:  " + results[i].getType());
                System.out.println("Area: x = " + results[i].getArea().x
                                      + " y = " + results[i].getArea().y
                                  + " width = " + results[i].getArea().width
                                 + " height = " + results[i].getArea().height);
                Point[] points = results[i].getPoints();
                System.out.format("Corners: Top Left(%d, %d), Top Right(%d, %d),"
                                + " Bottom Right(%d, %d), Bottom Left(%d, %d)\n",
                                  points[0].x, points[0].y, points[1].x, points[1].y,
                                  points[2].x, points[2].y, points[3].x, points[3].y);
                System.out.println("Confidence: " + results[i].getConfidence());
                System.out.println("Skew: " + results[i].getSkew());
                System.out.println();
            }
        }
        catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    public static String[] parseCommandLineArgs(String[] args, BarcodeReader reader) {
        List<BarcodeType> barcodeTypes = new ArrayList<>();

        int numOptions;
        for(numOptions = 0; numOptions < args.length; ++numOptions) {
            String arg0 = args[numOptions].toLowerCase();

            // No more options
            if (arg0.charAt(0) != '-')
                break;

            if (arg0.equals("-h") || arg0.equals("--help"))
                printUsage();

            if (arg0.equals("-t") || arg0.equals("--type")) {
                if (numOptions + 1 == args.length)
                    printUsage();

                numOptions++;
                String arg1 = args[numOptions].toLowerCase();
                if (arg1.equals("1d")) {
                    barcodeTypes.add(BarcodeType.ALL1D);
                } else if (arg1.equals("2d")) {
                    barcodeTypes.add(BarcodeType.ALL2D);
                } else if (arg1.equals("all")) {
                    barcodeTypes.add(BarcodeType.ALL);
                } else {
                    try {
                        barcodeTypes.add(BarcodeType.valueOf(arg1.toUpperCase()));
                    }
                    catch (IllegalArgumentException e){
                        System.out.println("Failed to parse barcode type: " + arg1);
                    }
                }
            }
        }

        if (barcodeTypes.isEmpty()) {
            // Add default barcode search type if user provided no custom type
            barcodeTypes.add(BarcodeType.ALL1D);
        }

        // Set scan barcode types
        reader.setBarcodeTypes(barcodeTypes.toArray(new BarcodeType[barcodeTypes.size()]));
        System.out.println("Searching for barcode types: " + barcodeTypes.toString());

        // No files in command line
        if (numOptions == args.length)
            printUsage();

        String[] fileNames = new String[args.length - numOptions];
        for (int i = 0; i < fileNames.length; i++) {
            fileNames[i] = args[numOptions + i];
        }

        return fileNames;
    }

    public static void printUsage() {
        System.out.println();
        System.out.println("Usage:");
        System.out.println("    ReadBarcodes [options] <image1> [<image2> ... <imageN>]");
        System.out.println();
        System.out.println("Options:");
        System.out.println("    -t, --type <type>       (optional = true, default = '1d') Barcode types to scan, where <type> represents:");
        System.out.println();
        System.out.println("                            Combinations of barcode types:");
        System.out.println("                                '1d' - all 1D barcodes");
        System.out.println("                                '2d' - all 2D barcodes");
        System.out.println("                                'all' - all 1D + 2D barcodes");
        System.out.println();
        System.out.println("                            Individual barcode types:");
        for (BarcodeType type : BarcodeType.values()) {
            if (type != BarcodeType.ALL && type != BarcodeType.ALL1D
                    && type != BarcodeType.ALL2D && type != BarcodeType.UNKNOWN) {
                System.out.println("                                " + type);
            }
        }
        System.out.println();
        System.out.println("    -h, --help              (optional = true) Print this help message");
        System.out.println();
        System.out.println("Examples:");
        System.out.println("    ReadBarcodes ../images/Barcode-All-Supported-Types.bmp ../images/Barcode-Multiple-Common.bmp");
        System.out.println("    ReadBarcodes -t 2d ../images/Barcode-All-Supported-Types.bmp");
        System.out.println("    ReadBarcodes -t 1d -t Planet --type ADD2 ../images/Barcode-All-Supported-Types.bmp");
        System.out.println("    ReadBarcodes --type all ../images/Barcode-*");
        System.exit(-1);
    }
}
